/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

import com.openexchange.mobile.api.facade.connectors.responses.mtos.ThreadReferenceMto;
import com.openexchange.mobile.api.facade.models.Attachment;
import com.openexchange.mobile.api.facade.models.Attachment.Disposition;
import com.openexchange.mobile.api.facade.models.Mail;
import com.openexchange.mobile.api.facade.models.MailContact;
import com.openexchange.mobile.api.facade.models.MailThread;

import lombok.experimental.UtilityClass;

@UtilityClass
public class MailUtil {

    private final MapFunction<ThreadReferenceMto, MailThread> threadMapFunction = new MapFunction<ThreadReferenceMto, MailThread>() {
        @Override
        public MailThread map(ThreadReferenceMto value) {
            return new MailThread(value);
        }
    };

    public Attachment getBody(Mail mail) {
        List<Attachment> parts = mail.getParts();
        Attachment contentPart = findContentPart(parts, "text/html");
        if (contentPart == null) {
            contentPart = findContentPart(parts, "text/plain");
        }
        return contentPart;
    }

    private Attachment findContentPart(List<Attachment> parts, String contentType) {
        if (parts == null) {
            return null;
        }
        for (Attachment part : parts) {
            Disposition partDisposition = part.getDisposition();
            String partContentType = part.getContentType();
            if (!Disposition.ATTACHMENT.equals(partDisposition)
                && !Disposition.NONE.equals(partDisposition)
                && contentType.equals(partContentType)) {
                return part;
            }
        }
        return null;
    }

    public boolean isTextPlain(Attachment part) {
        return part != null && "text/plain".equals(part.getContentType());
    }

    public List<MailThread> calculateThreads(List<Mail> mails) {
        if (mails == null) {
            return Collections.emptyList();
        }

        int index = 0;
        List<MailThread> threads = new ArrayList<>();
        List<Mail> threadMails = new ArrayList<>();
        Mail latestMail = null;

        while (index < mails.size()) {
            Mail mail = mails.get(index);

            if (mail.getLevel() == 0) {
                addThread(threads, threadMails, latestMail);
                threadMails = new ArrayList<>();
                latestMail = mail;
            } else if (latestMail == null || latestMail.getReceivedDate() < mail.getReceivedDate()) {
                latestMail = mail;
            }

            threadMails.add(mail);
            index++;
        }

        addThread(threads, threadMails, latestMail);

        return threads;
    }

    private void addThread(List<MailThread> threads, List<Mail> mails, Mail latestMail) {
        if (mails.size() > 0) {
            Mail rootMail = mails.get(0);
            threads.add(new MailThread(mails, rootMail.getId(), latestMail.getId(), latestMail.getReceivedDate()));
        }
    }

    public List<MailThread> wrapAsThreads(List<Mail> mails) {
        if (mails == null) {
            return Collections.emptyList();
        }
        List<MailThread> threads = new ArrayList<>(mails.size());
        for (Mail mail : mails) {
            List<Mail> threadMails = Collections.singletonList(mail);
            MailThread thread = new MailThread(threadMails, mail.getId(), mail.getId(), mail.getReceivedDate());
            threads.add(thread);
        }
        return threads;
    }

    public List<MailThread> convert(List<ThreadReferenceMto> data) {
        if (data == null) {
            return Collections.emptyList();
        }
        return ListUtil.map(data, threadMapFunction);
    }
    public List<MailContact> parseContacts(String value) {
        if (StringUtils.isEmpty(value)) {
            return null;
        }
        List<MailContact> contacts = new ArrayList<>(1);
        StringTokenizer tokenizer = new StringTokenizer(value, ";");
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            int indexLessThan = token.lastIndexOf('<');
            int indexGreaterThan = token.lastIndexOf('>');
            String name;
            String address;
            if (indexLessThan != -1 && indexGreaterThan != -1) {
                name = token.substring(0,  indexLessThan).trim();
                name = name.length() != 0 ? name : null;
                address = token.substring(indexLessThan + 1, indexGreaterThan).trim();
            } else {
                name = null;
                address = token.trim();
            }
            contacts.add(new MailContact(name, address));
        }
        return contacts;
    }

    public List<MailThread> removeDeletedMails(List<MailThread> mailThreads) {
        if (ListUtil.isEmpty(mailThreads)) {
            return Collections.emptyList();
        }
        List<MailThread> cleanedThreads = new ArrayList<>(mailThreads.size());
        for (MailThread mailThread : mailThreads) {
            List<Mail> mails = mailThread.getMails();
            List<Mail> nonDeletedMails = new ArrayList<>(mails.size());
            for (Mail mail : mails) {
                if (!mail.getFlags().isDeleted()) {
                    nonDeletedMails.add(mail);
                }
            }
            if (nonDeletedMails.size() > 0) {
                Mail rootMail = nonDeletedMails.get(0);
                Mail latestMail = nonDeletedMails.get(nonDeletedMails.size() - 1);
                cleanedThreads.add(new MailThread(nonDeletedMails, rootMail.getId(), latestMail.getId(), latestMail.getReceivedDate()));
            }
        }
        return cleanedThreads;
    }

}
