/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.connectors.impl;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.BaseConnector;
import com.openexchange.mobile.api.facade.connectors.RequestConfiguration;
import com.openexchange.mobile.api.facade.connectors.requests.mtos.MailRequestMto;
import com.openexchange.mobile.api.facade.connectors.responses.ArrayDataResponseMto;
import com.openexchange.mobile.api.facade.models.Mail;
import com.openexchange.mobile.api.facade.models.MailContact;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MailContactUtil;
import com.openexchange.mobile.api.facade.utils.MailUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.SneakyThrows;

import org.apache.http.HttpEntity;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.entity.StringEntity;

public class EnvelopesConnector extends BaseConnector<ArrayDataResponseMto> {

    private final String folderId;

    private final List<String> mailIds;

    private final MapFunction<String, MailRequestMto> mailMapFunction = new MapFunction<String, MailRequestMto>() {
        @Override
        public MailRequestMto map(String mailId) {
            return new MailRequestMto(folderId, mailId);
        }
    };

    public EnvelopesConnector(RequestConfiguration configuration, SessionData sessionData, String folderId, List<String> mailIds) {
        super(configuration, sessionData, "mail", ArrayDataResponseMto.class);
        this.folderId = folderId;
        this.mailIds = mailIds;
    }

    @Override
    protected HttpUriRequest createRequest() {
        RequestBuilder builder = RequestBuilder.put(createUrl());
        builder.setCharset(Charset.forName("UTF-8"));
        addSessionData(builder);
        builder.addParameter("action", "list");
        builder.addParameter("columns", "600,611,102,610,603,604,605,606,607,609,662,614,602");
        builder.addParameter("timezone", "utc");
        builder.setEntity(createEntity());
        return builder.build();
    }

    @SneakyThrows
    private HttpEntity createEntity() {
        /* Java8: Use with Java 8.
        List<MailRequestMto> mailRequestMtos = mailIds.stream()
                .map(mailId -> new MailRequestMto(folderId, mailId))
                .collect(Collectors.toList());
        */
        List<MailRequestMto> mailRequestMtos = ListUtil.map(mailIds, mailMapFunction);
        ObjectMapper mapper = JacksonUtil.getMapper();
        String str = mapper.writeValueAsString(mailRequestMtos);
        return new StringEntity(str, "UTF-8");
    }

    public static List<Mail> map(Object[][] data) {
        List<Mail> mails = new ArrayList<>();
        for (Object[] envelopData : data) {
            mails.add(EnvelopesConnector.map(envelopData));
        }
        return mails;
    }

    @SuppressWarnings("unchecked")
    public static Mail map(Object[] data) {
        String id = (String) data[0];
        Integer flags = (Integer) data[1];
        Integer color = (Integer) data[2];
        Long receivedDate = MailUtil.readAsLong(data[3]);
        List<MailContact> from = MailContactUtil.convert((List<List<String>>) data[4]);
        List<MailContact> to = MailContactUtil.convert((List<List<String>>) data[5]);
        List<MailContact> cc = MailContactUtil.convert((List<List<String>>) data[6]);
        List<MailContact> bcc = MailContactUtil.convert((List<List<String>>) data[7]);
        String subject = (String) data[8];
        Long sentDate = MailUtil.readAsLong(data[9]);
        String teaser = (String) data[10];
        Integer priority = (Integer) data[11];
        Boolean hasAttachments = (Boolean) data[12];
        return Mail.createEnvelope(id, flags, color, receivedDate, from, to, cc, bcc, subject, teaser, sentDate, priority, hasAttachments);
    }

}
