/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.openexchange.mobile.api.facade.models.Attachment;
import com.openexchange.mobile.api.facade.models.Attachment.Disposition;
import com.openexchange.mobile.api.facade.models.Mail;

import lombok.experimental.UtilityClass;

@UtilityClass
public class MailTransformations {

    private static final String REGEXP_TEXT_PLAIN_QUOTED = "^&gt;( |$)";

    private static final Pattern PATTERN_TEXT_PLAIN_QUOTED = Pattern.compile(REGEXP_TEXT_PLAIN_QUOTED);

    private static final String BLOCKQUOTE_START = "<blockquote type=\"cite\">";

    private static final String BLOCKQUOTE_END = "</blockquote>";

    public Mail apply(Mail mail, boolean isMailTeasersEnabled, int maximumTeaserLength) {
        combineContentParts(mail);
        Attachment part = MailUtil.getBody(mail);
        if (part == null) {
            return mail;
        }
        String content = part.getContent();
        content = transformContentSource(content);
        if (MailUtil.isTextPlain(part)) {
            content = transformTextContent(content);
        }
        part.setContent(content);
        if (isMailTeasersEnabled) {
            mail.setTeaser(getTeaser(mail, maximumTeaserLength));
        }
        return mail;
    }

    private void combineContentParts(Mail mail) {
        List<Attachment> parts = mail.getParts();
        String contentType = "text/html";
        List<Attachment> contentParts = findContentParts(parts, contentType);
        if (ListUtil.isEmpty(contentParts)) {
            contentType = "text/plain";
            contentParts = findContentParts(parts, contentType);
        }
        if (!ListUtil.isEmpty(contentParts) && contentParts.size() > 1) {
            parts = ListUtil.remove(parts, contentParts);
            String content = combineContentParts(contentParts);
            Attachment part = new Attachment("generated", Disposition.INLINE, contentType, content.length(), content, null, null, null, null);
            parts.add(0, part);
        } else if (!ListUtil.isEmpty(contentParts) && contentParts.size() == 1) {
            Attachment part = contentParts.get(0);
            if (parts.indexOf(part) > 0) {
                parts.remove(part);
                parts.add(0, part);
            }
        }
        mail.setParts(parts);
    }

    private String combineContentParts(List<Attachment> parts) {
        StringBuilder sb = new StringBuilder();
        for (Attachment part : parts) {
            String content = part.getContent();
            if (content != null) {
                sb.append(content);
            }
        }
        return sb.toString();
    }

    private List<Attachment> findContentParts(List<Attachment> parts, String contentType) {
        if (parts == null) {
            return null;
        }
        List<Attachment> contentParts = new ArrayList<>();
        for (Attachment part : parts) {
            Disposition partDisposition = part.getDisposition();
            String partContentType = part.getContentType();
            if (!Disposition.ATTACHMENT.equals(partDisposition)
                && !Disposition.NONE.equals(partDisposition)
                && contentType.equals(partContentType)) {
                contentParts.add(part);
            }
        }
        return contentParts;
    }

    public String transformContentSource(String content) {
        if (content == null) {
            return null;
        }
        content = content
                // Remove wbr tags.
                .replaceAll("<wbr>", "")
                // Remove leading white-space.
                .replaceAll("^(<div[^>]*>)(\\s|&nbsp;|<br ?\\/?>|<p[^>]*>(\\s|<br ?\\/?>|&nbsp;|&#160;)*<\\/p>|<div[^>]*>(\\s|<br\\/?>|&nbsp;|&#160;)*<\\/div>)+", "$1")
                // Remove closing <html> tag.
                .replaceAll("\\s*<\\/html>\\s*$", "")
                // Remove tailing white-space.
                .replaceAll("(\\s|&nbsp;|<br ?\\/?>|<p[^>]*>(\\s|<br\\/?>|&nbsp;|&#160;)*<\\/p>|<div[^>]*>(\\s|<br\\/?>|&nbsp;|&#160;)*<\\/div>)+<\\/div>$", "")
                // Fix Jericho broken links.
                .replaceAll("(<a href=\"https?://[^\"]+\" target=\"_blank\">https?://[^<]+</a>) &lt;<a href=\"https?://[^\"]+\" target=\"_blank\">https?://[^<]+</a>&gt;", "$1")
                ;
        return content;
    }

    public String transformTextContent(String content) {
        if (content == null) {
            return null;
        }
        content =
                // Replace plain text quotes with blockquotes.
                replacePlainTextQuotesWithBlockquotes(content)
                // Remove line breaks.
                .replaceAll("[\\r\\n]", "")
                // Remove leading BR.
                .replaceAll("^\\s*(<br\\s*\\/?>\\s*)+", "")
                // Reduce long BR sequences.
                .replaceAll("(<br\\/?>\\s*){3,}", "<br><br>")
                // combine split block quotes
                .replaceAll("<\\/blockquote>\\s*(<br *\\/?>\\s*)*<blockquote[^>]*>", "<br><br>")
                ;
        return content;
    }

    private String replacePlainTextQuotesWithBlockquotes(String content) {
        int level = 0;
        String[] lines = content.split("<[bB][rR]\\s*\\/?>");
        for (int index = 0, size = lines.length; index < size; index++) {
            String line = lines[index];
            if (line.length() != 0) {
                int lineLevel = 0;
                for (; PATTERN_TEXT_PLAIN_QUOTED.matcher(line).find(); lineLevel++) {
                    line = line.replaceFirst(REGEXP_TEXT_PLAIN_QUOTED, "");
                }
                for (; lineLevel > level; level++) {
                    line = BLOCKQUOTE_START + line;
                }
                for (; lineLevel < level; level--) {
                    lines[index - 1] = lines[index - 1] + BLOCKQUOTE_END;
                }
            }
            lines[index] = line;
        }
        content = join(lines);
        content = content.replaceAll("<br></blockquote>", "</blockquote>");
        return content;
    }

    private String join(String[] lines) {
        StringBuilder sb = new StringBuilder(4096);
        for (int index = 0, size = lines.length; index < size; index++) {
            if (index > 0) {
                sb.append("<br>");
            }
            sb.append(lines[index]);
        }
        return sb.toString();
    }

    private String getTeaser(Mail mail, int maximumTeaserLength) {
        String teaser = null;
        Attachment attachment = MailUtil.getBody(mail);
        String content = attachment != null ? attachment.getContent() : null;
        if (content != null) {
            Pattern headPattern = Pattern.compile("<head[^>]*>.*?</head>", Pattern.DOTALL);
            Matcher matcher = headPattern.matcher(content);
            teaser = matcher
                    .replaceFirst("")
                    .replaceAll("&#160;", "")
                    .replaceAll("&nbsp;", " ")
                    .replaceAll("(?s)<style.*?>.*?</style>", "")
                    .replaceAll("</?!?[a-zA-Z\\[\\]-]+[^>]*>", " ")
                    .replaceAll("\\s+", " ")
                    /* Java8: Use this with Java 8.
                    .replaceAll("(^\\h*)|(\\h*$)","")
                    */
                    .replaceAll("(^[ \\t\\xA0\\u1680\\u180e\\u2000-\\u200a\\u202f\\u205f\\u3000]*)|([ \\t\\xA0\\u1680\\u180e\\u2000-\\u200a\\u202f\\u205f\\u3000]*$)","")
                    .trim();
            teaser = HtmlUtil.unescapeHtml(teaser);
            if (teaser.length() > maximumTeaserLength) {
                teaser = teaser.substring(0, maximumTeaserLength)
                        .trim();
            }
        }
        return teaser;
    }

}
