/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.admin.reseller.console;

import java.util.HashSet;
import com.openexchange.admin.console.AdminParser;
import com.openexchange.admin.reseller.rmi.OXResellerInterface;
import com.openexchange.admin.reseller.rmi.OXResellerTools;
import com.openexchange.admin.reseller.rmi.dataobjects.ResellerAdmin;
import com.openexchange.admin.reseller.rmi.dataobjects.Restriction;
import com.openexchange.admin.rmi.dataobjects.Credentials;

/**
 * {@link Change} - changereseller command line tool
 *
 * @author choeger
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class Change extends ResellerAbstraction {

    /**
     * Entry point
     *
     * @param args command line arguments
     */
    public static void main(String[] args) {
        new Change().start(args);
    }

    /**
     * Initializes a new {@link Change}.
     */
    private Change() {
        super();
    }

    /**
     * Start the clt
     *
     * @param args the command line arguments
     */
    private void start(String[] args) {
        AdminParser parser = new AdminParser("changeadmin");

        setOptions(parser);

        String successtext = null;
        // parse the command line
        try {
            parser.ownparse(args);

            Credentials auth = credentialsparsing(parser);
            ResellerAdmin adm = parseChangeOptions(parser);

            OXResellerInterface rsi = getResellerInterface();

            parseAndSetAdminId(parser, adm);
            parseAndSetAdminname(parser, adm);

            successtext = nameOrIdSetInt(this.adminid, this.adminname, "admin");

            // Capabilities
            adm.setCapabilitiesToAdd(parseAndSetCapabilitiesToAdd(parser));
            adm.setCapabilitiesToRemove(parseAndSetCapabilitiesToRemove(parser));
            adm.setCapabilitiesToDrop(parseAndSetCapabilitiesToDrop(parser));

            // Configuration & Taxonomies
            applyDynamicOptionsToReseller(parser, adm);

            HashSet<String> removeRes = getRestrictionsToRemove(parser, this.removeRestrictionsOption);
            HashSet<Restriction> editRes = getRestrictionsToEdit(parser, this.editRestrictionsOption);
            ResellerAdmin dbadm = rsi.getData(adm, auth);
            HashSet<Restriction> dbres = OXResellerTools.array2HashSet(dbadm.getRestrictions());
            HashSet<Restriction> retRestrictions = handleAddEditRemoveRestrictions(dbres, OXResellerTools.array2HashSet(adm.getRestrictions()), removeRes, editRes);
            if (null != retRestrictions) {
                adm.setRestrictions(retRestrictions.toArray(new Restriction[retRestrictions.size()]));
            }
            rsi.change(adm, auth);
            displayChangedMessage(successtext, null, parser);
            sysexit(0);
        } catch (Exception e) {
            printErrors(successtext, null, e, parser);
            sysexit(1);
        }
    }

    /**
     * Sets the extra options for the clt
     *
     * @param parser The {@link AdminParser}
     */
    private void setOptions(AdminParser parser) {
        setChangeOptions(parser);
        parser.allowDynamicOptions(dynamicAttrDesc, dynamicRemoveAttrDesc);
        parser.allowFlexibleDynamicOptions();
    }
}
