/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.pns.transport.fcm.osgi;

import static com.openexchange.osgi.Tools.withRanking;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import org.osgi.framework.ServiceRegistration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.DefaultInterests;
import com.openexchange.config.ForcedReloadable;
import com.openexchange.config.Interests;
import com.openexchange.config.Reloadable;
import com.openexchange.config.cascade.ConfigViewFactory;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.DefaultUpdateTaskProviderService;
import com.openexchange.groupware.update.UpdateTaskProviderService;
import com.openexchange.java.Strings;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.pns.PushExceptionCodes;
import com.openexchange.pns.PushMessageGeneratorRegistry;
import com.openexchange.pns.PushSubscriptionRegistry;
import com.openexchange.pns.transport.fcm.DefaultFCMOptionsProvider;
import com.openexchange.pns.transport.fcm.FCMOptions;
import com.openexchange.pns.transport.fcm.FCMOptionsProvider;
import com.openexchange.pns.transport.fcm.groupware.RenameGCM2FCMUpdateTask;
import com.openexchange.pns.transport.fcm.internal.FCMPushNotificationTransport;

/**
 * {@link FCMPushNotificationTransportActivator}
 *
 * @author <a href="mailto:thorben.betten@open-xchange.com">Thorben Betten</a>
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @since v7.8.3
 */
public class FCMPushNotificationTransportActivator extends HousekeepingActivator implements Reloadable {

    private static final Logger LOG = LoggerFactory.getLogger(FCMPushNotificationTransportActivator.class);

    private static final String OPTIONS_FILENAME = "pns-fcm-options.yml";

    private ServiceRegistration<FCMOptionsProvider> optionsProviderRegistration;
    private FCMPushNotificationTransport fcmTransport;

    /**
     * Initializes a new {@link ApnPushNotificationTransportActivator}.
     */
    public FCMPushNotificationTransportActivator() {
        super();
    }

    @Override
    protected boolean stopOnServiceUnavailability() {
        return true;
    }

    @Override
    protected Class<?>[] getNeededServices() {
        // @formatter:off
        return new Class<?>[] { ConfigurationService.class,
                                PushSubscriptionRegistry.class,
                                PushMessageGeneratorRegistry.class,
                                ConfigViewFactory.class };
        // @formatter:on
    }

    @Override
    protected synchronized void startBundle() throws Exception {
        reinit(getService(ConfigurationService.class));
        registerService(UpdateTaskProviderService.class, new DefaultUpdateTaskProviderService(new RenameGCM2FCMUpdateTask()));
        registerService(ForcedReloadable.class, configService -> FCMPushNotificationTransport.invalidateEnabledCache());
        registerService(Reloadable.class, this);
    }

    /////////////////////////////// RELOADABLE STUFF ///////////////////////////////

    @Override
    public void reloadConfiguration(ConfigurationService configService) {
        try {
            reinit(configService);
        } catch (Exception e) {
            LOG.error("Failed to re-initialize FCM transport", e);
        }
    }

    @Override
    public Interests getInterests() {
        //@formatter:off
        return DefaultInterests.builder()
            .configFileNames(OPTIONS_FILENAME)
            .propertiesOfInterest("com.openexchange.pns.transport.fcm.enabled")
            .build();
        //@formatter:on
    }

    /**
     * Re-initialises the transport
     * 
     * @param configService The configuration service
     * @throws OXException if any error is occurred
     */
    private synchronized void reinit(ConfigurationService configService) throws OXException {
        FCMPushNotificationTransport transport = this.fcmTransport;
        if (null != transport) {
            transport.close();
            this.fcmTransport = null;
        }

        parseYAML(configService);

        transport = new FCMPushNotificationTransport(getService(PushSubscriptionRegistry.class), getService(PushMessageGeneratorRegistry.class), getService(ConfigViewFactory.class), context);
        transport.open();
        this.fcmTransport = transport;
    }

    /**
     * Parses the YAML config file
     * 
     * @param configService The config service
     * @throws OXException if an error is occurred
     */
    @SuppressWarnings("unchecked")
    private void parseYAML(ConfigurationService configService) throws OXException {
        ServiceRegistration<FCMOptionsProvider> optionsProviderReg = this.optionsProviderRegistration;
        if (null != optionsProviderReg) {
            optionsProviderReg.unregister();
            this.optionsProviderRegistration = null;
        }

        Object yaml = configService.getYaml(OPTIONS_FILENAME);
        if (!(yaml instanceof Map)) {
            return;
        }
        Map<String, Object> map = (Map<String, Object>) yaml;
        if (map.isEmpty()) {
            return;
        }
        Map<String, FCMOptions> options = parseFCMOptions(map);
        if (options.isEmpty()) {
            return;
        }
        optionsProviderReg = context.registerService(FCMOptionsProvider.class, new DefaultFCMOptionsProvider(options), withRanking(785));
        this.optionsProviderRegistration = optionsProviderReg;
    }

    /**
     * Parses the FCM options from the specified YAML map
     * 
     * @param yaml The YAML map
     * @return the options
     * @throws OXException if an error is occurred
     */
    private Map<String, FCMOptions> parseFCMOptions(Map<String, Object> yaml) throws OXException {
        Map<String, FCMOptions> options = new LinkedHashMap<>(yaml.size());
        for (Map.Entry<String, Object> entry : yaml.entrySet()) {
            process(options, entry);
        }
        return options;

    }

    /**
     * Processes the entry
     * 
     * @param options The options to put the entry after processing
     * @param entry The entry to process
     * @throws OXException if the configuration is invalid
     */
    @SuppressWarnings("unchecked")
    private void process(Map<String, FCMOptions> options, Entry<String, Object> entry) throws OXException {
        String client = entry.getKey();

        // Check for duplicate
        if (options.containsKey(client)) {
            throw PushExceptionCodes.UNEXPECTED_ERROR.create("Duplicate FCM options specified for client: " + client);
        }

        // Check values map
        if (!(entry.getValue() instanceof Map)) {
            throw PushExceptionCodes.UNEXPECTED_ERROR.create("Invalid FCM options configuration specified for client: " + client);
        }

        // Parse values map
        Map<String, Object> values = (Map<String, Object>) entry.getValue();

        // Enabled?
        Boolean enabled = getBooleanOption("enabled", Boolean.TRUE, values);
        if (!enabled.booleanValue()) {
            LOG.info("FCM options for client {} is disabled.", client);
            return;
        }
        String keyPath = getStringOption("keyPath", values);
        if (null == keyPath) {
            LOG.info("Missing 'keyPath' FCM option for client {}. Ignoring that client's configuration.", client);
            return;
        }
        FCMOptions fcmOptions = new FCMOptions(client, keyPath);
        options.put(client, fcmOptions);
        LOG.info("Parsed FCM options for client {}.", client);
    }

    /**
     * Retrieves a boolean options
     * 
     * @param name the name
     * @param def The default value
     * @param values The values
     * @return The boolean value
     */
    private Boolean getBooleanOption(String name, Boolean def, Map<String, Object> values) {
        Object object = values.get(name);
        if (object instanceof Boolean) {
            return (Boolean) object;
        }
        return null == object ? def : Boolean.valueOf(object.toString());
    }

    /**
     * Retrieves a string options
     * 
     * @param name the name
     * @param def The default value
     * @param values The values
     * @return The string value
     */
    private String getStringOption(String name, Map<String, Object> values) {
        Object object = values.get(name);
        if (null == object) {
            return null;
        }
        String str = object.toString();
        return Strings.isEmpty(str) ? null : str.trim();
    }
}
