/*
 * @copyright Copyright (c) Open-Xchange GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite. If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.office.tools.database;

import java.util.Properties;

/**
 * {@link DatabaseConnectionData}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.1
 */
public interface DatabaseConnectionData {

    /**
     * DEFAULT_DRIVER_CLASS
     */
    String DEFAULT_DRIVER_CLASSNAME = "com.mysql.jdbc.Driver";

    /**
     * DEFAULT_CONNECT_TIMEOUT_MILLIS
     */
    int DEFAULT_CONNECT_TIMEOUT_MILLIS = 60000;

    /**
     * DEFAULT_SOCKET_TIMEOUT_MILLIS
     */
    int DEFAULT_SOCKET_TIMEOUT_MILLIS = 60000;

    /**
     * @return The driver class for the database connection to be used.
     */
    default String getDriverClassName() {
        return DEFAULT_DRIVER_CLASSNAME;
    }

    /**
     * @return The URL for the database connection to be used.
     */
    String getConnectionURL();

    /**
     * @return The database username for the connection.
     */
    String getUserName();

    /**
     * @return The database password for the connection.
     */
    String getPassword();

    /**
     * @return Database specific JDBC {@link Properties}, containing
     *  {@link String} keys and {@link String}/{@link Boolean}/{@link Integer}
     *  object values throughout.
     */
    default Properties getJDBCProperties() {
        final Properties defaultDatabaseProperties = new Properties();

        defaultDatabaseProperties.put("useUnicode", "true");
        defaultDatabaseProperties.put("characterEncoding", "UTF-8");
        defaultDatabaseProperties.put("autoReconnect", "false");
        defaultDatabaseProperties.put("useServerPrepStmts", "true");
        defaultDatabaseProperties.put("useTimezone", "true");
        defaultDatabaseProperties.put("serverTimezone", "UTC");
        defaultDatabaseProperties.put("connectTimeout", Integer.valueOf(DEFAULT_CONNECT_TIMEOUT_MILLIS).toString());
        defaultDatabaseProperties.put("socketTimeout", Integer.valueOf(DEFAULT_SOCKET_TIMEOUT_MILLIS).toString());
        defaultDatabaseProperties.put("prepStmtCacheSize", Integer.valueOf(250).toString());
        defaultDatabaseProperties.put("prepStmtCacheSqlLimit", Integer.valueOf(4096).toString());
        defaultDatabaseProperties.put("cachePrepStmts", "true");
        defaultDatabaseProperties.put("useSSL", "false");
        defaultDatabaseProperties.put("requireSSL", "false");
        defaultDatabaseProperties.put("verifyServerCertificate", "false");
        defaultDatabaseProperties.put("enabledTLSProtocols", "TLSv1,TLSv1.1,TLSv1.2");
        defaultDatabaseProperties.put("clientCertificateKeyStoreUrl", "");
        defaultDatabaseProperties.put("clientCertificateKeyStoreType", "");
        defaultDatabaseProperties.put("clientCertificateKeyStorePassword", "");
        defaultDatabaseProperties.put("trustCertificateKeyStoreUrl", "");
        defaultDatabaseProperties.put("trustCertificateKeyStoreType", "");
        defaultDatabaseProperties.put("trustCertificateKeyStorePassword", "");

        return defaultDatabaseProperties;
    }

}
